params <-
list(css = "css/rmdformats.css")

## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.align = 'center'
)

## ----echo = FALSE, out.width = "100%", fig.cap = "Fig 1: f~up~ UC experimental set up", fig.topcaption = TRUE, fig.align = "center"----
knitr::include_graphics("img/fupUC_diagram.png")

## ----setup, message = FALSE, warning = FALSE----------------------------------
# Primary package 
library(invitroTKstats)
# Data manipulation package 
library(dplyr)
# Table formatting package 
library(flextable)

## ----Load example data--------------------------------------------------------
# Load example fup UC data 
data("Fup-UC-example")

## ----echo = FALSE, warning = FALSE--------------------------------------------
head(fup_uc_L0, n = 3) %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
        
      )
    )
  ) %>% 
  set_caption(caption = "Table 1: Level 0 data", 
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

## ----required cols, echo = FALSE----------------------------------------------
# Create table of required arguments for Level 1 

req_cols <- data.frame(matrix(nrow = 29, ncol = 5))
vars <- c("Argument", "Default", "Required in L0?", "Corresp. single-entry Argument", "Descr.")
colnames(req_cols) <- vars

# Argument names 
arguments <- c("FILENAME", "data.in", "sample.col", "lab.compound.col", "dtxsid.col", 
               "date.col", "compound.col", "area.col", "type.col", "test.conc.col", 
               "cal.col", "dilution.col", "istd.col", "istd.name.col", "istd.conc.col", 
               "uc.assay.conc.col", "biological.replicates.col",
               "technical.replicates.col", "analysis.method.col", 
               "analysis.instrument.col", "analysis.parameters.col", "note.col", 
               "level0.file.col", "level0.sheet.col", "output.res", "save.bad.types", 
               "sig.figs", "INPUT.DIR", "OUTPUT.DIR"
               )
req_cols[, "Argument"] <- arguments 

# Default arguments 
defaults <- c("MYDATA", NA, "Lab.Sample.Name", "Lab.Compound.Name", "DTXSID", 
              "Date", "Compound.Name", "Area", "Sample.Type", "Test.Compound.Conc", 
              "Cal", "Dilution.Factor", "ISTD.Area", "ISTD.Name", "ISTD.Conc", 
              "UC.Assay.Conc", "Biological.Replicates", "Technical.Replicates", 
              "Analysis.Method", "Analysis.Instrument", "Analysis.Parameters", 
              "Note", "Level0.File", "Level0.Sheet", FALSE, FALSE, 5, NA, NA)
req_cols[, "Default"] <- defaults 

# Argument required in L0?
req_cols <- req_cols %>% 
  mutate("Required in L0?" = case_when(
    Argument %in% c("sample.col", "lab.compound.col", "dtxsid.col",
                    "compound.col", "area.col", "type.col", 
                    "istd.col") ~ "Y", 
    Argument %in% c("FILENAME", "data.in", "output.res", "save.bad.types", 
                    "sig.figs", "INPUT.DIR", "OUTPUT.DIR") ~ "N/A",
    .default = "N"
  ))

# Corresponding single-entry Argument 
req_cols <- req_cols %>% 
  mutate("Corresp. single-entry Argument" = ifelse(.data[[vars[3]]] == "N" & .data[[vars[[1]]]] != "note.col",
                                                   gsub(".col", "", Argument), NA))

# Brief description 
description <- c("Output and input filename", 
                 "Level 0 data frame", 
                 "Lab sample name", 
                 "Lab test compound name (abbr.)", 
                 "EPA's DSSTox Structure ID",
                 "Lab measurement date", 
                 "Formal test compound name", 
                 "Target analyte peak area", 
                 "Sample type (CC/AF/T1/T5)",
                 "Standard test chemical concentration", 
                 "MS calibration", 
                 "Number of times sample was diluted", 
                 "Internal standard peak area", 
                 "Internal standard name", 
                 "Internal standard concentration", 
                 "Intended initial test concentration",
                 "Replicates with the same analyte", 
                 "Repeated measurements from one sample", 
                 "Analytical chemistry analysis method", 
                 "Analytical chemistry analysis instrument", 
                 "Analytical chemistry analysis parameters", 
                 "Additional notes", 
                 "Raw data filename", 
                 "Raw data sheet name", 
                 "Export results (TSV)?", 
                 "Export bad data (TSV)?",
                 "Number of significant figures",
                 "Input directory of Level 0 file", 
                 "Export directory to save Level 1 files")
req_cols[, "Descr."] <- description

## ----echo = FALSE, warning = FALSE--------------------------------------------
req_cols %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
        height = 200
      )
    )
  ) %>% 
  set_caption(caption = "Table 2: Level 1 `format_fup_uc` function arguments", align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

## ----L1 processing------------------------------------------------------------
fup_uc_L1_curated <- format_fup_uc(FILENAME = "Fup_UC_vignette",
                                   data.in = fup_uc_L0,
                                   # columns present in L0 data
                                   sample.col = "Sample",
                                   lab.compound.col = "Lab.Compound.ID",
                                   compound.col = "Compound",
                                   area.col = "Peak.Area", 
                                   test.conc.col = "Compound.Conc",
                                   cal.col = "Date", 
                                   istd.col = "ISTD.Peak.Area",
                                   technical.replicates.col = "Replicate",
                                   analysis.parameters.col = "Analysis.Params",
                                   # columns not present in L0 data 
                                   istd.conc = 1, 
                                   test.nominal.conc = 10, 
                                   biological.replicates = 1, 
                                   analysis.method = "UPLC-MS/MS",
                                   analysis.instrument = "Waters Xevo TQ-S micro(QEB0036)",
                                   note.col = NULL,
                                   # don't export output TSV file
                                   output.res = FALSE
                                   )

## ----echo = FALSE, warning = FALSE--------------------------------------------
fup_uc_L1_curated %>% 
  head(n = 3) %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
      )
      )
  ) %>% 
  set_caption(caption = "Table 3: Level 1 data",
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

## -----------------------------------------------------------------------------
# Use verification data from loaded in `fup_uc_L2` data frame 
exclusion <- fup_uc_L2 %>% 
  filter(Verified != "Y") %>% 
  mutate("Variables" = "Lab.Sample.Name|DTXSID") %>% 
  mutate("Values" = paste(Lab.Sample.Name, DTXSID, sep = "|")) %>% 
  mutate("Message" = Verified) %>% 
  select(Variables, Values, Message)

## ----echo = FALSE-------------------------------------------------------------
exclusion %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list()
      )
  ) %>% 
  set_caption("Table 4: Exclusion data", align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "body") %>% 
  theme_vanilla()

## ----L2 processing------------------------------------------------------------
fup_uc_L2_curated <- sample_verification(FILENAME = "fup_UC_vignette", 
                                         data.in = fup_uc_L1_curated,
                                         assay = "fup-UC",
                                         # don't export output TSV file
                                         output.res = FALSE)

## ----echo = FALSE-------------------------------------------------------------
fup_uc_L2_curated %>% 
  head(n = 3) %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list()
      )
  ) %>% 
  set_caption("Table 5: Level 2 data", align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "body") %>% 
  theme_vanilla()

## ----L3 processing------------------------------------------------------------
fup_uc_L3_curated <- calc_fup_uc_point(FILENAME = "Fup_UC_vignette", 
                                 data.in = fup_uc_L2_curated,
                                 # don't export output TSV file 
                                 output.res = FALSE)

## ----echo = FALSE-------------------------------------------------------------
fup_uc_L3_curated %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_caption("Table 6: Level 3 data", align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "body") %>% 
  theme_vanilla()

## ----L4 processing, message = FALSE, eval = FALSE-----------------------------
# fup_uc_L4_curated <- calc_fup_uc(FILENAME = "Fup_UC_vignette",
#                                  data.in = fup_uc_L2_curated,
#                                  JAGS.PATH = runjags::findjags()
#                                  )

## ----echo = FALSE-------------------------------------------------------------
fup_uc_L4_curated <- fup_uc_L4

## ----echo = FALSE-------------------------------------------------------------
fup_uc_L4_curated %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
      )
      )
  ) %>% 
  set_caption(caption = "Table 7: Level 4 data",
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

